
import sys
import time
import threading
import numpy as np
import cv2
import os

def resource_path(relative_path):
    base_path = getattr(sys, '_MEIPASS', os.path.abspath("."))
    return os.path.join(base_path, relative_path)

from collections import deque
from datetime import datetime
from ffpyplayer.player import MediaPlayer
from PyQt5.QtWidgets import QWidget, QLabel, QVBoxLayout, QHBoxLayout, QPushButton, QSizePolicy, QMessageBox
from PyQt5.QtGui import QPixmap, QFont, QImage
from PyQt5.QtCore import Qt, QSettings, QTimer
from buffer_manager import BufferManager
from ui.analise_window import AnaliseWindow

class VideoWindow(QWidget):
    def __init__(self, cam1_config, cam2_config, delay, area_name, num_cameras=2):
        super().__init__()
        self.setWindowTitle("VillaReplay")
        self.setWindowState(Qt.WindowMaximized)

        self.settings = QSettings("VillaReplay", "Config")
        self.theme = self.settings.value("theme", "light")
        self.apply_theme()

        self.delay = delay
        self.area_name = area_name
        self.num_cameras = num_cameras
        self.fps = 15

        self.buffer_cam1 = BufferManager(max_seconds=30, fps=self.fps)
        self.queue1 = deque(maxlen=delay * self.fps + 30)

        if self.num_cameras == 2:
            self.buffer_cam2 = BufferManager(max_seconds=30, fps=self.fps)
            self.queue2 = deque(maxlen=delay * self.fps + 30)

        self.cam1_url = f"rtsp://{cam1_config['user']}:{cam1_config['pass']}@{cam1_config['ip']}/cam/realmonitor?channel={cam1_config['channel']}&subtype=0"
        if self.num_cameras == 2:
            self.cam2_url = f"rtsp://{cam2_config['user']}:{cam2_config['pass']}@{cam2_config['ip']}/cam/realmonitor?channel={cam2_config['channel']}&subtype=0"

        self.init_ui()
        self.start_players()
        self.start_timer()

    def apply_theme(self):
        if self.theme == "dark":
            self.setStyleSheet("background-color: #0B132B; color: white;")
        else:
            self.setStyleSheet("background-color: #f0f0f0; color: black;")

    def toggle_theme(self):
        self.theme = "dark" if self.theme == "light" else "light"
        self.settings.setValue("theme", self.theme)
        self.apply_theme()

    def init_ui(self):
        self.logo = QLabel()
        self.logo.setAlignment(Qt.AlignCenter)
        custom_logo_path = resource_path("custom_logo.png")
        if os.path.exists(custom_logo_path):
            pix = QPixmap(custom_logo_path)
        else:
            pix = QPixmap(resource_path("logo.png"))

        if not pix.isNull():
            self.logo.setPixmap(pix.scaledToHeight(127, Qt.SmoothTransformation))
        else:
            self.logo.setText("VillaReplay")
            self.logo.setFont(QFont("Arial", 20, QFont.Bold))

        self.area_label = QLabel(self.area_name)
        self.area_label.setAlignment(Qt.AlignRight)
        self.area_label.setFont(QFont("Arial", 20, QFont.Bold))
        self.area_label.setStyleSheet("padding: 6px;")

        self.btn_config = QPushButton("⚙️")
        self.btn_config.setFixedSize(36, 36)
        self.btn_config.clicked.connect(lambda: self.config_callback() if hasattr(self, 'config_callback') else None)

        self.btn_about = QPushButton("ℹ️")
        self.btn_about.setFixedSize(36, 36)
        self.btn_about.clicked.connect(self.mostrar_sobre)

        self.btn_theme = QPushButton("🌓")
        self.btn_theme.setFixedSize(36, 36)
        self.btn_theme.clicked.connect(self.toggle_theme)

        self.label_cam1 = QLabel()
        self.label_cam1.setAlignment(Qt.AlignCenter)
        self.label_cam1.setStyleSheet("background-color: black")
        self.label_cam1.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

        layout_videos = QHBoxLayout()
        layout_videos.setContentsMargins(10, 0, 10, 0)
        layout_videos.setSpacing(10)
        layout_videos.addWidget(self.label_cam1)

        if self.num_cameras == 2:
            self.label_cam2 = QLabel()
            self.label_cam2.setAlignment(Qt.AlignCenter)
            self.label_cam2.setStyleSheet("background-color: black")
            self.label_cam2.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
            layout_videos.addWidget(self.label_cam2)

        self.btn_analise = QPushButton("📁 Analisar Vídeo")
        self.btn_analise.setCursor(Qt.PointingHandCursor)
        self.btn_analise.setFixedSize(220, 50)
        self.btn_analise.setFont(QFont("Arial", 13))
        self.btn_analise.setStyleSheet("QPushButton { background-color: #4682B4; color: white; border-radius: 10px; } QPushButton:hover { background-color: #5a9bd5; }")
        self.btn_analise.clicked.connect(self.abrir_analise)

        self.logo_villa = QLabel()
        pix_footer = QPixmap(resource_path("logo.png"))
        if not pix_footer.isNull():
            self.logo_villa.setPixmap(pix_footer.scaledToWidth(80, Qt.SmoothTransformation))

        bottom_right_layout = QHBoxLayout()
        bottom_right_layout.addWidget(self.btn_about)
        bottom_right_layout.addWidget(self.btn_theme)
        bottom_right_layout.addWidget(self.btn_config)
        bottom_right_layout.addWidget(self.logo_villa)
        bottom_right_layout.setAlignment(Qt.AlignRight)
        bottom_right_layout.setSpacing(10)

        bottom_layout = QHBoxLayout()
        bottom_layout.addWidget(self.btn_analise, alignment=Qt.AlignLeft)
        bottom_layout.addStretch()
        bottom_layout.addLayout(bottom_right_layout)

        layout = QVBoxLayout()
        layout.addWidget(self.logo)
        layout.addWidget(self.area_label)
        layout.addLayout(layout_videos)
        layout.addLayout(bottom_layout)
        self.setLayout(layout)

    def start_players(self):
        threading.Thread(target=self.player_thread, args=(self.cam1_url, self.queue1, self.buffer_cam1), daemon=True).start()
        if self.num_cameras == 2:
            threading.Thread(target=self.player_thread, args=(self.cam2_url, self.queue2, self.buffer_cam2), daemon=True).start()

    def player_thread(self, url, queue, buffer):
        player = MediaPlayer(url, ff_opts={
            "rtsp_transport": "tcp",
            "fflags": "nobuffer",
            "flags": "low_delay",
            "framedrop": "0",
            "infbuf": "1",
            "probesize": "1000000",
            "an": True
        })
        while True:
            frame, val = player.get_frame()
            if val == 'eof' or frame is None:
                time.sleep(0.005)
                continue

            img = frame[0]
            if img is None:
                time.sleep(0.005)
                continue

            try:
                w, h = img.get_size()
                bdata = img.to_bytearray()[0]
                arr = np.frombuffer(bdata, dtype=np.uint8).reshape((h, w, 3))
                arr = cv2.cvtColor(arr, cv2.COLOR_RGB2BGR)
                queue.append((time.time(), arr))
                buffer.add_frame(arr)
            except Exception as e:
                print(f"[ERRO ao decodificar imagem] {e}")
            time.sleep(1 / self.fps)

    def start_timer(self):
        self.timer = QTimer()
        self.timer.timeout.connect(self.display_delayed_frames)
        self.timer.start(int(1000 / self.fps))

    def display_delayed_frames(self):
        now = time.time()
        tdelay = self.delay
        frame1 = self.get_delayed_frame(self.queue1, now - tdelay)
        if frame1 is not None:
            self.show_frame(frame1, self.label_cam1)
        if self.num_cameras == 2:
            frame2 = self.get_delayed_frame(self.queue2, now - tdelay)
            if frame2 is not None:
                self.show_frame(frame2, self.label_cam2)

    def get_delayed_frame(self, queue, target_time):
        for ts, frame in reversed(queue):
            if ts <= target_time:
                return frame
        return None

    def show_frame(self, frame, label):
        rgb = cv2.cvtColor(frame, cv2.COLOR_BGR2RGB)
        h, w, ch = rgb.shape
        img = QImage(rgb.data, w, h, ch * w, QImage.Format_RGB888)
        pixmap = QPixmap.fromImage(img)
        label.setPixmap(pixmap.scaled(label.width(), label.height(), Qt.KeepAspectRatio))

    def abrir_analise(self):
        timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
        os.makedirs("analises", exist_ok=True)
        arq1 = f"analises/{self.area_name}_Cam1_{timestamp}.avi"
        self.buffer_cam1.save_to_file(arq1)
        if self.num_cameras == 2:
            arq2 = f"analises/{self.area_name}_Cam2_{timestamp}.avi"
            self.buffer_cam2.save_to_file(arq2)
            self.janela_analise = AnaliseWindow(arq1, arq2)
        else:
            self.janela_analise = AnaliseWindow(arq1)
        self.janela_analise.show()

    def mostrar_sobre(self):
        about = QMessageBox(self)
        about.setWindowTitle("Sobre o VillaReplay")
        about.setText(
            "VillaReplay - Sistema de Vídeo Replay para Arbitragem de Judô\n"
            "Versão: V.1.1.0\n\n"
            "Desenvolvido por Luiz Emilio Villanueva\n"
            "Contato: contato@villareplay.com.br"
        )
        about.setStyleSheet("QLabel{color:black;} QMessageBox{background-color:#e0e0e0;}")
        about.exec_()

    def set_config_callback(self, callback):
        self.config_callback = callback
