
import shutil
import json
import os
import requests
import sys
from PyQt5.QtWidgets import (
    QWidget, QLabel, QLineEdit, QPushButton, QVBoxLayout,
    QFormLayout, QSpinBox, QMessageBox, QComboBox, QFileDialog, QHBoxLayout)
from PyQt5.QtCore import QTimer
from ui.video_window import VideoWindow

CONFIG_FILE = "villa_replay_config.json"
CACHE_FILE = "chave_autorizada.json"

def resource_path(relative_path):
    base_path = getattr(sys, '_MEIPASS', os.path.abspath("."))
    return os.path.join(base_path, relative_path)

class ConfigWindow(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("VillaReplay - Configuração Inicial")
        self.setGeometry(100, 100, 400, 560)
        self.init_ui()
        self.load_config()

    def init_ui(self):
        layout = QVBoxLayout()
        self.form_layout = QFormLayout()

        self.area_name = QLineEdit()
        self.form_layout.addRow(QLabel("Nome da Área:"), self.area_name)

        self.key_input = QLineEdit()
        self.key_input.setPlaceholderText("Digite sua chave de ativação")
        self.form_layout.addRow(QLabel("🔑 Chave de ativação:"), self.key_input)

        self.cam_count = QComboBox()
        self.cam_count.addItems(["1", "2"])
        self.cam_count.currentIndexChanged.connect(self.update_cam_visibility)
        self.form_layout.addRow(QLabel("Número de Câmeras:"), self.cam_count)

        self.cam1_ip = QLineEdit()
        self.cam1_user = QLineEdit()
        self.cam1_pass = QLineEdit()
        self.cam1_channel = QSpinBox()
        self.cam1_channel.setRange(1, 8)
        self.cam1_pass.setEchoMode(QLineEdit.Password)

        self.form_layout.addRow(QLabel("Câmera 1 - IP:"), self.cam1_ip)
        self.form_layout.addRow(QLabel("Usuário:"), self.cam1_user)
        self.form_layout.addRow(QLabel("Senha:"), self.cam1_pass)
        self.form_layout.addRow(QLabel("Canal:"), self.cam1_channel)

        self.cam2_ip = QLineEdit()
        self.cam2_user = QLineEdit()
        self.cam2_pass = QLineEdit()
        self.cam2_channel = QSpinBox()
        self.cam2_channel.setRange(1, 16)
        self.cam2_pass.setEchoMode(QLineEdit.Password)

        self.cam2_widgets = [
            (QLabel("Câmera 2 - IP:"), self.cam2_ip),
            (QLabel("Usuário:"), self.cam2_user),
            (QLabel("Senha:"), self.cam2_pass),
            (QLabel("Canal:"), self.cam2_channel)
        ]
        for label, widget in self.cam2_widgets:
            self.form_layout.addRow(label, widget)

        self.delay_spin = QSpinBox()
        self.delay_spin.setRange(1, 60)
        self.delay_spin.setValue(5)
        self.delay_spin.setSuffix(" segundos")
        self.form_layout.addRow(QLabel("Atraso padrão:"), self.delay_spin)

        self.logo_path = QLineEdit()
        self.logo_path.setReadOnly(True)

        self.logo_btn = QPushButton("Selecionar Logo Personalizada")
        self.logo_btn.clicked.connect(self.selecionar_logo)

        self.remove_logo_btn = QPushButton("Remover Logo")
        self.remove_logo_btn.clicked.connect(self.remover_logo)

        logo_btn_layout = QHBoxLayout()
        logo_btn_layout.addWidget(self.logo_btn)
        logo_btn_layout.addWidget(self.remove_logo_btn)

        self.form_layout.addRow(QLabel("Logo personalizada:"), logo_btn_layout)
        self.form_layout.addRow(self.logo_path)

        self.start_button = QPushButton("Iniciar ou Aplicar Configurações")
        self.start_button.clicked.connect(self.apply_or_launch)

        layout.addLayout(self.form_layout)
        layout.addWidget(self.start_button)
        self.setLayout(layout)

        self.update_cam_visibility()

    def update_cam_visibility(self):
        show_cam2 = self.cam_count.currentText() == "2"
        for label, widget in self.cam2_widgets:
            label.setVisible(show_cam2)
            widget.setVisible(show_cam2)

    def load_config(self):
        try:
            with open(CONFIG_FILE, "r") as f:
                config = json.load(f)
                self.area_name.setText(config.get("area_name", "Área 01"))
                self.key_input.setText(config.get("chave", ""))
                self.cam_count.setCurrentIndex(int(config.get("num_cameras", 2)) - 1)

                self.cam1_ip.setText(config.get("cam1_ip", ""))
                self.cam1_user.setText(config.get("cam1_user", ""))
                self.cam1_pass.setText(config.get("cam1_pass", ""))
                self.cam1_channel.setValue(config.get("cam1_channel", 1))

                self.cam2_ip.setText(config.get("cam2_ip", ""))
                self.cam2_user.setText(config.get("cam2_user", ""))
                self.cam2_pass.setText(config.get("cam2_pass", ""))
                self.cam2_channel.setValue(config.get("cam2_channel", 1))

                self.delay_spin.setValue(config.get("delay", 5))
                self.logo_path.setText(config.get("logo_path", ""))
        except FileNotFoundError:
            pass

    def save_config(self):
        config = {
            "area_name": self.area_name.text(),
            "chave": self.key_input.text(),
            "num_cameras": int(self.cam_count.currentText()),

            "cam1_ip": self.cam1_ip.text(),
            "cam1_user": self.cam1_user.text(),
            "cam1_pass": self.cam1_pass.text(),
            "cam1_channel": self.cam1_channel.value(),

            "cam2_ip": self.cam2_ip.text(),
            "cam2_user": self.cam2_user.text(),
            "cam2_pass": self.cam2_pass.text(),
            "cam2_channel": self.cam2_channel.value(),

            "delay": self.delay_spin.value(),
            "logo_path": self.logo_path.text()
        }
        with open(CONFIG_FILE, "w") as f:
            json.dump(config, f, indent=4)

    def get_hardware_id(self):
        import uuid
        return str(uuid.getnode())

    def validar_chave_remota(self, chave):
        chave = chave.strip()
        hwid = self.get_hardware_id()
        try:
            url = f"http://www.villareplay.com.br/prog_control/valida.php?chave={chave}&hwid={hwid}"
            r = requests.get(url, timeout=5)
            resposta = r.text.strip().upper()
            if "OK" in resposta:
                with open(CACHE_FILE, "w") as f:
                    json.dump({"chave": chave, "hwid": hwid}, f)
                return True
            elif "EXPIRADA" in resposta:
                QMessageBox.warning(self, "Chave Expirada", "Sua chave de ativação expirou.")
                return False
            elif "HWID_INVALIDO" in resposta:
                QMessageBox.critical(self, "Acesso negado", "Esta chave já foi ativada em outro computador.")
                return False
        except:
            pass

        if os.path.exists(CACHE_FILE):
            try:
                with open(CACHE_FILE, "r") as f:
                    dados = json.load(f)
                    return dados.get("chave", "") == chave and dados.get("hwid", "") == hwid
            except:
                return False
        return False

    def apply_or_launch(self):
        self.save_config()

        chave = self.key_input.text().strip()
        if not self.validar_chave_remota(chave):
            QMessageBox.critical(self, "Erro", "Chave inválida ou conexão recusada.")
            return

        cam1_config = {
            'ip': self.cam1_ip.text(),
            'user': self.cam1_user.text(),
            'pass': self.cam1_pass.text(),
            'channel': self.cam1_channel.value()
        }
        cam2_config = {
            'ip': self.cam2_ip.text(),
            'user': self.cam2_user.text(),
            'pass': self.cam2_pass.text(),
            'channel': self.cam2_channel.value()
        }
        delay = self.delay_spin.value()
        area = self.area_name.text()
        num_cameras = int(self.cam_count.currentText())

        try:
            video = VideoWindow(cam1_config, cam2_config, delay, area, num_cameras)
            video.set_config_callback(self.show)
            video.show()
            self.video_window = video
            self.close()
        except Exception as e:
            print(f"Erro ao aplicar configurações: {e}")

    def selecionar_logo(self):
        path, _ = QFileDialog.getOpenFileName(self, "Selecionar Logo", "", "Imagens (*.png *.jpg *.jpeg)")
        if path:
            try:
                destino = resource_path("custom_logo.png")
                shutil.copy(path, destino)
                self.logo_path.setText("custom_logo.png")
            except Exception as e:
                QMessageBox.warning(self, "Erro", f"Erro ao copiar a imagem: {e}")

    def remover_logo(self):
        try:
            path = resource_path("custom_logo.png")
            if os.path.exists(path):
                os.remove(path)
            self.logo_path.setText("")
            QMessageBox.information(self, "Logo Removida", "Logo personalizada removida com sucesso.")
        except Exception as e:
            QMessageBox.warning(self, "Erro", f"Erro ao remover a logo: {e}")
